#include <cassert>
#include <cstdio>
#include <set>

#include "union_find_delete.h"

UnionFindDelete::UnionFindDelete(uint init_size) : 
  d(init_size), freed(init_size, 0), num_freed(init_size), \
  clist(init_size), nlist(init_size), dfslist(init_size), item2pd(init_size, UINT_NULL)
{
  uint i = 0;
  for (list<uint>::iterator it = freed.begin(); it != freed.end(); ++ it){
    *it = i;
    ++i;
  }
}

SimpleUnionFindDelete::SimpleUnionFindDelete(uint init_size) :
  pd2item(init_size), item2pd(init_size, UINT_NULL)
{}

ListBank::ListBank(uint init_size) :
  freed(init_size, 0), num_freed(init_size), \
  d(init_size)
{
  uint i = 0;
  for (list<uint>::iterator it = freed.begin(); it != freed.end(); ++ it){
    *it = i;
    ++i;
  }
}


ListListBank::ListListBank(uint init_size) :
  freed(init_size, 0), num_freed(init_size), \
  d(init_size)
{
  uint i = 0;
  for (list<uint>::iterator it = freed.begin(); it != freed.end(); ++ it){
    *it = i;
    ++i;
  }
}

uint ListBank::new_elem() {
  if (num_freed == 0) {
    // too small; double the size;
    uint s = d.size();
    d.insert(d.end(), s, ListElem());
    num_freed = s;
    for (uint i = s; i < s+s; ++i)
      freed.push_back(i);
  }

  --num_freed;
  uint p = freed.front();
  freed.pop_front();
  return p;
}

uint ListListBank::new_elem() {
  if (num_freed == 0) {
    // too small; double the size;
    uint s = d.size();
    d.insert(d.end(), s, list<uint>());
    num_freed = s;
    for (uint i = s; i < s+s; ++i)
      freed.push_back(i);
  }

  --num_freed;
  uint p = freed.front();
  freed.pop_front();
  return p;
}

bool ListBank::free_elem(uint k) {
  d[k].p_node = UINT_NULL; // mark it has been freed
  freed.push_front(k);
  ++num_freed;
  return true;
}

bool ListListBank::free_elem(uint k) {
  assert(d[k].size() == 0); // make sure there is no elements in it!
  freed.push_front(k);
  ++num_freed;
  return true;
}

uint UnionFindDelete::new_d() {
  if (num_freed == 0) {
    // too small; double the size;
    uint s = d.size();
    d.insert(d.end(), s, TreeNode());
    num_freed = s;
    for (uint i = s; i < s+s; ++i)
      freed.push_back(i);
  }

  --num_freed;
  uint p = freed.front();
  freed.pop_front();
  //printf("in new_d: num_freed=%u, p=%u\n", num_freed, p);
  return p;
}

bool UnionFindDelete::free_d(uint k) {
  d[k].p_parent = UINT_NULL; // mark it has been freed
  freed.push_front(k);
  ++num_freed;
  return true;
}

void UnionFindDelete::update_item2pd(uint item, uint pd) {
  if (item >= item2pd.size()){
    uint cs = item2pd.size();
    uint es = (item - cs + 1 > cs + cs ? item - cs + 1 : cs + cs);
    item2pd.insert(item2pd.end(), es, UINT_NULL);
  }
  item2pd[item] = pd;
}

void SimpleUnionFindDelete::update_item2pd(uint item, uint pd) {
  if (item >= item2pd.size()){
    uint cs = item2pd.size();
    uint es = (item - cs + 1 > cs + cs ? item - cs + 1 : cs + cs);
    item2pd.insert(item2pd.end(), es, UINT_NULL);
  }
  item2pd[item] = pd;
}

uint SimpleUnionFindDelete::makeset(uint item) {
  if (item < item2pd.size() && item2pd[item] != UINT_NULL) {
    // already created;
    return item2pd[item];
  }
  uint pd = pd2item.new_elem();
  pd2item.d[pd].push_front(item);
  update_item2pd(item, pd);
  return pd;
}

uint UnionFindDelete::makeset(uint item) {
  if (item < item2pd.size() && item2pd[item] != UINT_NULL) {
    // already created;
    return item2pd[item];
  }
  
  uint p = new_d();

  d[p].item = item;
  update_item2pd(item, p);
  
  d[p].p_parent = p;
  d[p].rank = 0;
  
  d[p].head_clist = UINT_NULL;
  d[p].head_nlist = UINT_NULL;
  d[p].p_clist = UINT_NULL;
  d[p].p_nlist = UINT_NULL;
  
  uint p_dfs = dfslist.new_elem();
  dfslist.d[p_dfs].p_node = p;
  dfslist.d[p_dfs].prev = p_dfs;
  dfslist.d[p_dfs].next = p_dfs;
  d[p].p_dfs = p_dfs;

  d[p].size = 1;
  
  return p;
}

void UnionFindDelete::print_node(uint p) {
  printf("NODE[pointer=%u]\n", p);
  printf("  item=%u\n", d[p].item);
  printf("  p_parent=%u\n", d[p].p_parent);
  printf("  rank=%u\n", d[p].rank);
  printf("  head_clist=%u\n", d[p].head_clist);
  printf("  head_nlist=%u\n", d[p].head_nlist);
  printf("  p_clist=%u\n", d[p].p_clist);
  printf("  p_nlist=%u\n", d[p].p_nlist);
  printf("  p_dfs=%u\n", d[p].p_dfs);
  printf("  size=%u\n", d[p].size);
}


void ListBank::append(uint q, uint& p) {
  if (p == UINT_NULL) {
    p = q;
    d[q].prev = q;
    d[q].next = q;
  }else{
    uint endp = d[p].prev;
    d[q].prev = endp;
    d[endp].next = q;
    d[q].next = p;
    d[p].prev = q;
  }
}

uint SimpleUnionFindDelete::union_(uint p_a, uint p_b) {
  if (p_a == p_b) // they are the same
    return p_a;
  if (pd2item.d[p_a].size() < pd2item.d[p_b].size()) {
    uint tmp = p_b;
    p_b = p_a;
    p_a = tmp;
  }
  for (list<uint>::iterator it = pd2item.d[p_b].begin(); it != pd2item.d[p_b].end(); ++ it) {
    pd2item.d[p_a].push_back(*it);
    update_item2pd(*it, p_a);
  }
  pd2item.d[p_b].clear();
  pd2item.free_elem(p_b);
  return p_a;
}

uint UnionFindDelete::union_(uint p_a, uint p_b) {
  assert(d[p_a].p_parent == p_a);  // check if they are all root nodes
  assert(d[p_b].p_parent == p_b);
  if (p_a == p_b) // they are the same
    return p_a;
  if (d[p_a].size < d[p_b].size) {
    uint tmp = p_b;
    p_b = p_a;
    p_a = tmp;
  }
  if (d[p_b].size < 4) {
    //printf("d[p_b].size < 4\n");
    // attach its children to d[p_a]
    uint pc = d[p_b].head_clist;
    if (pc != UINT_NULL) {
      uint pc_new = UINT_NULL;
      assert(d[clist.d[pc].p_node].p_nlist == UINT_NULL);
      add_single_node_no_clist_update(clist.d[pc].p_node, p_a);
      pc_new = clist.d[pc].next;
      clist.append(pc, d[p_a].head_clist);
      pc = pc_new;
      while (pc != d[p_b].head_clist) {
        assert(d[clist.d[pc].p_node].p_nlist == UINT_NULL);
        add_single_node_no_clist_update(clist.d[pc].p_node, p_a);
        pc_new = clist.d[pc].next;
        clist.append(pc, d[p_a].head_clist);
        pc = pc_new;
      }
    }
    // attach itself to d[p_a]
    assert(d[p_b].p_nlist == UINT_NULL);
    add_single_node_no_clist_update(p_b, p_a);
    pc = clist.new_elem();
    d[p_b].p_clist = pc;
    clist.d[pc].p_node = p_b;
    clist.append(pc, d[p_a].head_clist);
  }else{
    //printf("both tree's size > 4\n");
    //check_dfslist_one_root(p_a);
    //check_dfslist_one_root(p_b);
    // both tree's size > 4
    d[p_b].p_parent = p_a;
    if (d[p_a].rank < d[p_b].rank + 1)
      d[p_a].rank = d[p_b].rank + 1;
    // add p_b to the clist of p_a
    uint pc = clist.new_elem();
    d[p_b].p_clist = pc;
    clist.d[pc].p_node = p_b;
    clist.insert_first(pc, d[p_a].head_clist);
    // add p_b to the nlist of p_a
    uint pn = nlist.new_elem();
    d[p_b].p_nlist = pn;
    nlist.d[pn].p_node = p_b;
    nlist.insert_first(pn, d[p_a].head_nlist);
    // update dfs list
    dfslist.merge_after_first(d[p_b].p_dfs, d[p_a].p_dfs);
    // remove the nlist of p_b
    free_nlist(d[p_b].head_nlist);
    // update p_a's size
    d[p_a].size += d[p_b].size;
    //check_dfslist_one_root(p_a);
  }
  return p_a;
}

void UnionFindDelete::free_nlist(uint& pn_head) {
  uint pn = pn_head;
  uint pn_new = UINT_NULL;
  if (pn == UINT_NULL)
    return;  
  pn_new = nlist.d[pn].next;
  // delete from the node
  d[nlist.d[pn].p_node].p_nlist = UINT_NULL;
  // delete from the nlist
  nlist.free_elem(pn);
  pn = pn_new;
  while (pn != pn_head) {
    pn_new = nlist.d[pn].next;
    // delete from the node
    d[nlist.d[pn].p_node].p_nlist = UINT_NULL;
    // delete from the nlist
    nlist.free_elem(pn);
    pn = pn_new;
  }
  pn_head = UINT_NULL;
}


void ListBank::insert_first(uint q, uint& p) {
  if (p == UINT_NULL) {
    p = q;
    d[q].prev = q;
    d[q].next = q;    
  }else{
    d[q].prev = d[p].prev;
    d[d[p].prev].next = q;
    d[q].next = p;
    d[p].prev = q;
    p = q;
  }
}

void ListBank::merge_after_first(uint q, uint p) {
  assert(p != UINT_NULL);
  assert(q != UINT_NULL);
  
  uint q_last = d[q].prev;
  
  d[q_last].next = d[p].next;
  d[d[p].next].prev = q_last;

  d[p].next = q;
  d[q].prev = p;
}


void UnionFindDelete::add_single_node_no_clist_update(uint p, uint pf) {
  d[p].p_parent = pf;
  d[p].rank = 0;
  d[p].head_clist = UINT_NULL;
  d[p].head_nlist = UINT_NULL;
  d[p].p_nlist = UINT_NULL;

  dfslist.append(d[p].p_dfs, d[pf].p_dfs);

  d[pf].rank = (d[pf].rank > 1 ? d[pf].rank : 1);
  ++d[pf].size;
}


/*
inline bool UnionFindDelete::isroot(uint a) {
  return (d[a].p_parent == a);
}
*/

inline bool UnionFindDelete::has_left_sibling(uint x) {
  //uint chead = d[d[x].p_parent].head_clist;
  //return (clist.d[d[x].p_clist].next != chead);
  return (clist.d[d[x].p_clist].next != d[d[x].p_parent].head_clist);
}

inline bool UnionFindDelete::has_right_sibling(uint x) {
  //uint chead = d[d[x].p_parent].head_clist;
  //return (d[x].p_clist != chead);
  return (d[x].p_clist != d[d[x].p_parent].head_clist);
}

inline bool UnionFindDelete::isleaf(uint a) {
  return (d[a].head_clist == UINT_NULL);
}

inline uint UnionFindDelete::left_sibling(uint x) {
  return clist.d[clist.d[d[x].p_clist].next].p_node;
}


uint SimpleUnionFindDelete::find_(uint a) {
  return item2pd[a];
}

uint UnionFindDelete::find_(uint a) {
  //printf("in find %u: pa=%u, isroot=%d\n", a, d[a].p_parent, isroot(a));
  if (isroot(a))
    return a;
  else{
    uint t = d[a].p_parent;
    while (!isroot(t)) {
      relink(a, t, d[t].p_parent);
      a = t;
      t = d[a].p_parent;
    }
    return t;
  }
}

void UnionFindDelete::relink(uint a, uint p, uint gp) {
  //printf("relink(a=%u, p=%u, gp=%u)\n", a, p, gp);
  // relink to the grand parent
  if (has_left_sibling(a)) {
    //printf("in has_left_sibling\n");
    //check_dfslist_one_root(gp);
    
    uint l = left_sibling(a);

    /*printf("clist.move_out (relink): d[a].p_clist=%u, d[p].head_clist=%u", \
             d[a].p_clist, d[p].head_clist); 
    */
    //fflush(stdout);
    clist.move_out(d[a].p_clist, d[p].head_clist);
    //printf(" done.\n");
    //fflush(stdout);
    
    clist.insert_before(d[a].p_clist, d[p].p_clist, d[gp].head_clist);
    // update dfslist
    uint ld = d[l].p_dfs;
    uint pld = dfslist.d[ld].prev;
    uint pd = d[p].p_dfs;
    uint ppd = dfslist.d[pd].prev;
    uint ad = d[a].p_dfs;
    uint pad = dfslist.d[ad].prev;
    //printf("ld=%u pld=%u pd=%u ppd=%u ad=%u pad=%u\n", ld, pld, pd, ppd, ad, pad);
    // from       pp->p-->>pa->a-->>pl->l
    // change to  pp->a-->>pl->p-->>pa->l
    dfslist.d[ppd].next = ad;
    dfslist.d[ad].prev = ppd;
    dfslist.d[pld].next = pd;
    dfslist.d[pd].prev = pld;
    dfslist.d[pad].next = ld;
    dfslist.d[ld].prev = pad;

    d[a].p_parent = gp;
    
    if (!isleaf(a) && isroot(gp)) {
      uint pn = nlist.new_elem();
      d[a].p_nlist = pn;
      nlist.d[pn].p_node = a;
      nlist.insert_before(pn, d[p].p_nlist, d[gp].head_nlist);
    }
    //printf("after relink\n");
    //check_dfslist_one_root(gp);
  }else{
    /*printf("clist.move_out (relink): d[a].p_clist=%u, d[p].head_clist=%u", \
           d[a].p_clist, d[p].head_clist);
    fflush(stdout);
    */
    clist.move_out(d[a].p_clist, d[p].head_clist);
    //printf(" done.\n");
    //fflush(stdout);
    clist.insert_after(d[a].p_clist, d[p].p_clist);

    d[a].p_parent = gp;
    
    if (!isleaf(a) && isroot(gp)) {
      uint pn = nlist.new_elem();
      d[a].p_nlist = pn;
      nlist.d[pn].p_node = a;
      nlist.insert_after(pn, d[p].p_nlist);
    }
    // update dfslist; no need to update
    //printf("after relink\n");
    //check_dfslist_one_root(gp);
  }
  if (isleaf(p)) {
    d[p].rank = 0;
    if (isroot(gp)) {
      /*printf("nlist.move_out (relink): d[p].p_nlist=%u, d[gp].head_nlist=%u", \
               d[p].p_nlist, d[gp].head_nlist);
      */
      //fflush(stdout);
      nlist.move_out(d[p].p_nlist, d[gp].head_nlist);
      //printf(" done.\n");
      //fflush(stdout);
      nlist.free_elem(d[p].p_nlist);
      d[p].p_nlist = UINT_NULL;
      if (d[gp].head_nlist == UINT_NULL)
        d[gp].rank = 1;
    }
  }else{
    // check if p only has <= two off-springs, if it is the case, then relink them
    if (!more_offsprings(p, 2)){
      relink(clist.d[d[p].head_clist].p_node, p, gp);
    }
  }
}

bool UnionFindDelete::more_offsprings(uint p, int num) {
  if (num < 0)
    return true;
  uint hc = d[p].head_clist;
  if (hc == UINT_NULL)
    return false;
  else{
    uint c = hc;
    --num;
    while (num >= 0){
      c = clist.d[c].next;
      if (c == hc)
        break;
      --num;
    }
    if (num >= 0)
      return false;
    else
      return true;
  }
}

void ListBank::move_out(uint q, uint& p) {
  if (p == d[p].next){
    //
    if (q != p) {
      printf("Error:: ListBank::moveout q=%u, p=%u, d[p].prev=%u, d[p].next=%u\n", q, \
             p, d[p].prev, d[p].next);
      fflush(stdout);
    }      
    assert(q == p);
    p = UINT_NULL;
  }else if (q == p){
    p = d[q].next;
    uint r = d[q].prev;
    d[r].next = p;
    d[p].prev = r;
  }else{
    uint pp = d[q].next;
    uint r = d[q].prev;
    //printf("pp=%u r=%u\n", pp, r);
    d[r].next = pp;
    d[pp].prev = r;    
  }
}

void ListBank::insert_after(uint q, uint p) {
  uint r = d[p].next;
  d[p].next = q;
  d[q].prev = p;
  d[q].next = r;
  d[r].prev = q;
}


void ListBank::insert_before(uint q, uint p, uint& h) {
  if (p == h)
    h = q;
  uint r = d[p].prev;
  d[r].next = q;
  d[q].prev = r;
  d[q].next = p;
  d[p].prev = q;
}


uint SimpleUnionFindDelete::delete_(uint a, uint r) {
  if (pd2item.d[r].size() == 1)
    return r;

  list<uint>::iterator it = pd2item.d[r].begin();
  for ( ; it != pd2item.d[r].end(); ++ it)
    if ((*it) == a)
      break;
  assert( it != pd2item.d[r].end() );
  pd2item.d[r].erase(it);

  uint pd = pd2item.new_elem();
  pd2item.d[pd].push_front(a);
  update_item2pd(a, pd);

  return pd;
}

pair<uint, uint> UnionFindDelete::delete_(uint a, uint r) {
  //uint r = find_(a); // r is the root
  //printf("in delete\n");
  if (d[r].size <= 4)
    assert(d[r].rank <= 1);
  if (d[r].rank <= 1)
    return reduced_tree_delete(a, r);
  else if (isleaf(a))
    return delete_leaf(a, r, 1);
  else{
    uint l = find_leaf(a, r);
    assert(isleaf(l));
    // swap the content of a and l
    update_item2pd(d[l].item, a);
    update_item2pd(d[a].item, l);
    
    uint tmp = d[l].item;
    d[l].item = d[a].item;
    d[a].item = tmp;
    
    a = l;
    return delete_leaf(a, r, 1);
  }
}


pair<uint, uint> UnionFindDelete::reduced_tree_delete(uint a, uint r) {
  //printf("in reduced_tree_delete\n");
  if (isleaf(a))
    return delete_leaf(a, r, 0);
  else{
    uint ch = clist.d[d[a].head_clist].p_node;
    //printf("ch=%u a=%u r=%u d[ch].item=%u d[a].item=%u\n", ch, a, r, d[ch].item, d[a].item);
    /*
    if (!isleaf(ch)) {
      printf("reduced_tree_delete:: ch=%u a=%u r=%u d[ch].item=%u d[a].item=%u\n", ch, a, r, d[ch].item, d[a].item);
      check_clist_recursively(r, r);
    }
    */
    assert(isleaf(ch));
    // swap the content of a and ch
    update_item2pd(d[ch].item, a);
    update_item2pd(d[a].item, ch);
    
    uint tmp = d[ch].item;
    d[ch].item = d[a].item;
    d[a].item = tmp;
    a = ch;
    return delete_leaf(a, r, 0);
  }
}


void UnionFindDelete::print_one_layer_clist(uint a) {
    uint pc_head = d[a].head_clist;
    uint pc = pc_head;
    
    printf("  children:");
    fflush(stdout);
    do{
      uint p = clist.d[pc].p_node;
      printf(" [pc=%u, pc.prev=%u, pc.next=%u, p=%u, item=%u]", pc, clist.d[pc].prev, clist.d[pc].next, p, d[p].item);
      fflush(stdout);
      pc = clist.d[pc].next;
    }while (pc != pc_head);
    printf("\n");
}

void UnionFindDelete::print_dfslist(uint a) {
    uint pd_head = d[a].p_dfs;
    uint pd = pd_head;
    
    printf("  dfslist:");
    fflush(stdout);
    do{
      uint p = dfslist.d[pd].p_node;
      printf(" [pd=%u, pd.prev=%u, pd.next=%u, p=%u, item=%u]", pd, dfslist.d[pd].prev, dfslist.d[pd].next, p, d[p].item);
      fflush(stdout);
      pd = dfslist.d[pd].next;
    }while (pd != pd_head);
    printf("\n");
}


pair<uint, uint> UnionFindDelete::delete_leaf(uint a, uint r, uint option) {
  //printf("in delete_leaf a=%u, r=%u, option=%u\n", a, r, option);
  assert(isleaf(a));
  if (a == r) {
    return make_pair(a, a);
  }else{
    uint p = d[a].p_parent;

    d[a].p_parent = a;
    d[a].rank = 0;

    /*
    print_one_layer_clist(p);
    printf("  d[a].p_clist=%u d[p].head_clist=%u\n", d[a].p_clist, d[p].head_clist);
    fflush(stdout);
    */
    
    /*printf("clist.move_out (delete_leaf): d[a].p_clist=%u, d[p].head_clist=%u", \
             d[a].p_clist, d[p].head_clist);
    */
    //fflush(stdout);
    clist.move_out(d[a].p_clist, d[p].head_clist);
    //printf(" done\n");
    //fflush(stdout);
    
    clist.free_elem(d[a].p_clist);
    d[a].p_clist = UINT_NULL;

    /*
    if (d[p].head_clist != UINT_NULL) {
      printf("after move_out: ");
      print_one_layer_clist(p);
    }
    */

    if (d[p].head_clist == UINT_NULL){
      d[p].rank = 0;
      if (d[p].p_nlist != UINT_NULL) {
        uint gp = d[p].p_parent;
        nlist.move_out(d[p].p_nlist, d[gp].head_nlist);
        nlist.free_elem(d[p].p_nlist);
        d[p].p_nlist = UINT_NULL;
        if (d[gp].head_nlist == UINT_NULL)
          d[gp].rank = 1;
      }
    }
    
    //printf("dfslist.move_out: d[a].p_dfs=%u, d[r].p_dfs=%u\n", d[a].p_dfs, d[r].p_dfs);
    //print_dfslist(d[r].p_dfs);
    dfslist.move_out(d[a].p_dfs, d[r].p_dfs);

    //printf("done dfslist.move_out\n");
    //fflush(stdout);
    
    dfslist.d[d[a].p_dfs].prev = d[a].p_dfs;
    dfslist.d[d[a].p_dfs].next = d[a].p_dfs;

    assert(d[a].p_nlist == UINT_NULL);
    
    d[a].size = 1;
    --d[r].size;

    // local rebuild
    if (option > 0){ 
      if (p != r) {
        uint gp = d[p].p_parent;
        for (uint i = 0; i < 2; ++ i) {
          uint ch = clist.d[d[p].head_clist].p_node;
          relink(ch, p, gp);
          if (d[p].head_clist == UINT_NULL)
            break;
        }
      }else{
        uint c = nlist.d[d[r].head_nlist].p_node;
        for (uint i = 0; i < 3; ++ i) {
          uint ch = clist.d[d[c].head_clist].p_node;
          relink(ch, c, r);
          if (d[c].head_clist == UINT_NULL)
            break;
        }
      }
    }

    return make_pair(a, r);
  }
}


uint UnionFindDelete::find_leaf(uint a, uint r) {
  if (isleaf(a))
    return a;
  else if (a == r){
    uint prev = dfslist.d[d[a].p_dfs].prev;
    return dfslist.d[prev].p_node;
  }else{
    if (has_left_sibling(a)){
      uint ls = left_sibling(a);
      uint prev = dfslist.d[d[ls].p_dfs].prev;
      return dfslist.d[prev].p_node;
    }else if (has_right_sibling(a)){
      uint prev = dfslist.d[d[a].p_dfs].prev;
      return dfslist.d[prev].p_node;
    }else{
      assert(false);
    }
  }
}


void SimpleUnionFindDelete::print_set(uint p) {
  if (pd2item.d.size() <= p){
    printf("Error from print_set(%u): p=%u, pd2item.d.size()=%lu\n", p, p, \
           pd2item.d.size());
  }else{
    list<uint>::iterator it = pd2item.d[p].begin();
    printf("[ ");
    for (; it != pd2item.d[p].end(); ++ it)
      printf("%u ", *it);
    printf("]");
  }
}


bool UnionFindDelete::check_clist_recursively(uint a, uint r) {
  printf("check_clist_recursively(a=%u, a.item=%u, r=%u, r.item=%u) d[a].parent=%u\n",\
         a, d[a].item, r, d[r].item, d[a].p_parent);
  if (d[a].p_parent != r){
    printf("assert will fail!\n");
    fflush(stdout);
  }
  assert(d[a].p_parent == r);
  if (d[a].p_parent != r)
    return false;
  if (!isleaf(a)) {
    bool flag = true;
    uint pc_head = d[a].head_clist;
    uint pc = pc_head;
    
    printf("  children:");
    fflush(stdout);
    do{
      uint p = clist.d[pc].p_node;
      printf(" [p=%u, item=%u]", p, d[p].item);
      fflush(stdout);
      pc = clist.d[pc].next;
    }while (pc != pc_head);
    printf("\n");
    
    do{
      uint p = clist.d[pc].p_node;
      if (!check_clist_recursively(p, a))
        flag = false;
      pc = clist.d[pc].next;
    }while (pc != pc_head);
    return flag;
  }
  return true;
}


void UnionFindDelete::print_recursive_tree_items(uint a, uint r, FILE* oup) {
  
  if (d[a].p_parent != r){
    printf("Error in print_recursive_tree_items(a=%u, r=%u), a.parent=%u != r\n",\
           a, r, d[a].p_parent);
    fflush(stdout);    
  }
  assert(d[a].p_parent == r);

  if (a == r)
    fprintf(oup, "%u", d[a].item);
  else
    fprintf(oup, ", %u", d[a].item);

  if (!isleaf(a)) {
    uint pc_head = d[a].head_clist;
    uint pc = pc_head;
    
    do{
      uint p = clist.d[pc].p_node;
      print_recursive_tree_items(p, a, oup);
      pc = clist.d[pc].next;
    }while (pc != pc_head);
  }
}


bool UnionFindDelete::check_dfslist_one_root(uint root_pd) {
  printf("root_pd=%u size=%u ", root_pd, d[root_pd].size);
  fflush(stdout);
  uint p_dfs = d[root_pd].p_dfs;
  for (uint j = 0; j < d[root_pd].size; ++ j) {
    printf("[p_dfs=%u dfslist.d[p_dfs].p_node=%u next=%u prev=%u] ", p_dfs, \
           dfslist.d[p_dfs].p_node, dfslist.d[p_dfs].next, dfslist.d[p_dfs].prev);
    fflush(stdout);
    //assert(find_(dfslist.d[p_dfs].p_node) == root_pd);
    assert(dfslist.d[dfslist.d[p_dfs].next].prev == p_dfs);
    p_dfs = dfslist.d[p_dfs].next;
  }
  printf("\n");
  return true;
}

bool UnionFindDelete::check_from_dfslist(uint n_items) {
  set<uint> root_set;
  uint size = 0;
  for (uint i = 0; i < n_items; ++ i) {
    uint root_pd = find_(item2pd[i]);
    if (root_set.find(root_pd) == root_set.end()) {
      printf("root_pd=%u size=%u\n", root_pd, d[root_pd].size);
      //
      uint p_dfs = d[root_pd].p_dfs;
      for (uint j = 0; j < d[root_pd].size; ++ j) {
        //assert(find_(dfslist.d[p_dfs].p_node) == root_pd);
        assert(dfslist.d[dfslist.d[p_dfs].next].prev == p_dfs);
        p_dfs = dfslist.d[p_dfs].next;
      }
      assert(p_dfs == d[root_pd].p_dfs);
      root_set.insert(root_pd);
      //
      size += d[root_pd].size;
    }
  }
  printf("size=%u n_items=%u #(distinct_roots)=%lu\n", size, n_items, root_set.size());
  assert(size == n_items);
  return (size == n_items);
}
